<?php

namespace App\Repositories\Product;

use App\Models\File;
use App\Models\Product;
use App\Repositories\Shared\Infrastructure\Services\File\FileRepository;
use App\Repositories\Shared\Infrastructure\Services\Storage\StorageRepository;
use Illuminate\Support\Facades\Storage;

class ProductRepositoryImpl implements ProductRepository
{

    private StorageRepository $storageRepository;
    private FileRepository $fileRepository;

    public function __construct(Product $product, StorageRepository $storageRepository, FileRepository $fileRepository)
    {
        $this->storageRepository = $storageRepository;
        $this->fileRepository = $fileRepository;
    }

    public function getAll()
    {
        return Product::all();
    }

    public function paginate($limit = null, $columns = ['*'])
    {
        return Product::with(['TypeProduct', 'TypeSkins', 'Files'])->paginate($limit, $columns);
    }

    public function getById($id)
    {
        return Product::findOrFail($id);
    }

    public function delete($id): string
    {
        $product = Product::find($id);

        if (is_null($product)) {
            throw new \Exception('Product not found.');
        }

        $product->update(['active' => false]);

        $files = $product->files;

        foreach ($files as $file) {
            $this->storageRepository->delete($file->path, $file->namefile);
            $this->fileRepository->delete($file->id);
        }

        $product->delete();

        return 'successfully deleted';

    }

    public function create(array $details)
    {
        try {
            $product = Product::create($details);
            $product->typeskins()->attach($details['type_skin_id']);

            $path = 'products/';
            foreach ($details['files'] as $file) {
                // File
                $fileModel = $this->fileRepository->create($file, $path, $product->id, 'products');

                // Storage
                $filePath = $path . $fileModel->namefile;
                $this->storageRepository->put($file, $filePath);
            }

            $product->TypeSkins;
            $product->Files;
            $product->TypeProduct;

            return $product;

        } catch (\Exception $e) {
            throw new \Exception($e->getMessage());

        }
    }

    public function update($id, array $data): object
    {
        $typeSkins = $data['type_skin_id'];
        unset($data['type_skin_id']);

        $product = Product::find($id);

        if (is_null($product)) {
            throw new \Exception('Product not found.');
        }

        $updated = $product->update($data);
        $product->typeSkins()->sync($typeSkins);


        // delete Files
        if (isset($data['files_to_delete']) && count($data['files_to_delete'])) {
            
            $filesToDelete = File::where('table_name', 'products')
                ->where('table_id', $id)
                ->whereIn('id', $data['files_to_delete'])
                ->get();

            foreach ($filesToDelete as $file) {
                $this->deleteImage($file);
            }
        }


        // update files
        $path = 'products/';

        $filesOld = $this->fileRepository->getAllByModel($id, 'products');

        if (isset($data['files']) && count($data['files'])) {
            foreach ($data['files'] as $file) {
                $exist = false;

                foreach ($filesOld as $key => $old) {

                    if (!strcmp($old->checksum, hash_file('sha1', $file->getRealPath()))) {
                        $filesOld->forget($key);
                        $exist = true;
                        continue;
                    }
                }
                if ($exist) {
                    continue;
                }

                //create
                // File
                $fileModel = $this->fileRepository->create($file, $path, $product->id, 'products');
                $ids[] = $fileModel->id;

                // Storage
                $filePath = $path . $fileModel->namefile;
                $this->storageRepository->put($file, $filePath);
            }
        }

        // relationships
        $product->typeSkins;
        $product->TypeProduct;
        $product->Files;

        if ($updated == 0) {
            throw new \Exception('Product not found.');
        }

        return $product;
    }

    public function getImage($file)
    {
        try {
            if (!strcmp($file->table_name, 'products')) {
                $image = $this->storageRepository->getFileByFullName($file->path, $file->namefile); //Storage::get($file->path . $file->namefile);
                return response($image, 200)->header('Content-Type', Storage::mimeType($file->filetype));

            }

            return abort(404);

        } catch (\Exception $e) {
            throw new \Exception($e->getMessage());

        }

    }

    public function deleteImage($file)
    {
        try {

            if (!strcmp($file->table_name, 'products')) {
                $this->storageRepository->delete($file->path, $file->namefile);

                $this->fileRepository->delete($file->id);

                return true;

            }

            throw new \Exception('No es una imagen de producto');

        } catch (\Exception $e) {
            throw new \Exception($e->getMessage());

        }

    }

}
