<?php

namespace App\Repositories\Appointment;

use App\Models\Appointment;
use Carbon\Carbon;
use Carbon\CarbonPeriod;
use Carbon\CarbonInterval;
use App\Models\Service;
use Illuminate\Support\Facades\Auth;

class AppointmentRepositoryImpl implements AppointmentRepository
{
    private const MINUTES = 5;
    private const SLOT = self::MINUTES . ' minutes';

    public function paginate($limit = null, $filter = [], $columns = ['*'])
    {
        return Appointment::with(['client'])
            ->where(function ($query) use ($filter) {
                if (count($filter)) {
                    if (isset($filter['status'])) {
                        $query->where('status', $filter['status']);
                    }

                    if (isset($filter['client_id'])) {
                        $query->where('client_id', $filter['client_id']);
                    }

                }
            })->orderBy('created_at', 'DESC')
            ->paginate($limit, $columns);
    }

    public function getById($id)
    {
        return Appointment::findOrFail($id);
    }

    public function delete($id): string
    {
        $deleted = Appointment::find($id);

        if (is_null($deleted)) {
            throw new \Exception('Appointment not found.');
        }

        $deleted->update(['active' => false, 'deleted_by' => Auth::id()]);
        $deleted->delete();

        return 'successfully deleted';

    }

    public function create(array $details)
    {
        
        if(!Auth::user()->hasRole('admin')){
            $details['client_id'] = Auth::id();
        }
        $service = Service::find($details['service_id']);
        $start_time = Carbon::parse($details['start_time']);

        // validate available time
        $slots = $this->appointmentAvailable(['date' => $start_time->format('Y-m-d'), 'service_id' => $details['service_id']]);
          
        if(!isset($slots['free_slots']) || (count($slots['free_slots'])==0))
            throw new \Exception('No hay disponibilidad de horarios.');
        
        if(!in_array($start_time->format('H:i'), $slots['free_slots']))
            throw new \Exception('Horario no disponible.');
        // *********

        $duration= Carbon::parse($service->duration);
        
        if(!is_null($service->additional_time)){
            $additional_time = Carbon::parse($service->additional_time);
             $duration->addHour($additional_time->format('H'))->addMinutes($additional_time->format('i'));
        }

        $finish_time = $start_time;
        $finish_time->addHour($duration->format('H'))->addMinutes($duration->format('si'));
        $details['finish_time'] = $finish_time;
        
        return Appointment::create($details);
    }

    public function update($id, array $data): string
    {
        $updated = Appointment::whereId($id)->update($data);

        if ($updated == 0) {
            throw new \Exception('Appointment not found.');
        }

        return 'successfully updated';
    }

    public function appointmentAvailable($queries): array
    {
        $date = $queries['date'];//Carbon::today()->toDateString();

        $startHour = Carbon::parse($date . ' ' . env('WORK_START_TIME'));
        $endHour = Carbon::parse($date . ' ' . env('WORK_END_TIME'));

        $bookedSlots = $this->getDateAppointments($date);

        $slotsService = $this->getSlotsService($queries['service_id']);

        $allSlots = CarbonPeriod::create($startHour, self::SLOT, $endHour)
            ->excludeEndDate()
            ->filter(function ($time) use ($bookedSlots) {
                return !in_array($time->format('H:i'), $bookedSlots);
            });

        $allSlotsHours = [];
        foreach ($allSlots as $slot) {
            $allSlotsHours[] = $slot->format('H:i');
        }
        
        $freeSlots = [];

        foreach ($allSlots as $k => $slot) {
            $slotsToUse = $this->generateSlotsToUse($slot, $slotsService);
            if ($this->isSubset($allSlotsHours, $slotsToUse)) {
                $freeSlots[] = $allSlotsHours[$k];
            }
        }

        $data = [];
        $filters = [];
        
        foreach($freeSlots as $slot){
            $filter = substr($slot, 0, -2);
            // dump($filter);
            if(!in_array($filter, $filters)){
                $filters[]= $filter;
                // $data[intval(str_replace(':','',$filter))] = array_values($this->order($freeSlots, $filter));
                $data[] = ['hour' => intval(str_replace(':','',$filter)), 'minutes' => array_values($this->order($freeSlots, $filter))];

            }
        }
        // return $data;
        return [
            'free_slots'=> $freeSlots,
             'times' => $data
        ];
    }

    private function order($array, $val)
    {
        $filtered = array_filter($array, function ($v) use ($val) { 
            return str_starts_with($v,$val);
        });

        $result = str_replace($val, '', $filtered);
  
        return $result;
    }

    private function getSlotsService($serviceId)
    {   
        $service = Service::find($serviceId);

        $start= Carbon::parse('00:00')->format('H:i');
        $duration = Carbon::parse($service->duration);//->format('H:i');

        $additional_time = Carbon::parse($service->additional_time);
        $total_minutes = $additional_time->diffInMinutes($start);
        $additional_time =  $duration->addMinutes($total_minutes);
        $slots = CarbonPeriod::create($start, self::SLOT, $duration)->excludeEndDate();
        return $slots->count();

    }

    private function isSubset(array $arr1, array $arr2): bool
    {
        return count(array_intersect($arr2, $arr1)) == count($arr2);
    }

    private function generateSlotsToUse($date, $slot = 0)
    {

        $dates = [];
        $dates[] = $date->format('H:i');

        for ($i = 1; $i < $slot; $i++) {
            $dates[] = $date->addMinutes(self::MINUTES)->format('H:i');
        }

        return $dates;
    }

    private function getDateAppointments($date)
    {
        $bookedSlots = [];
        // $today = Carbon::now()->format('Y-m-d') . '%';
        $appointmens = Appointment::where('start_time', 'LIKE', $date . '%')->get();

        foreach ($appointmens as $appointment) {
            $bookedSlots = array_merge($this->getBookingByAppointment($appointment), $bookedSlots);
        }

        return $bookedSlots;

    }

    private function getBookingByAppointment($appointment)
    {

        $period = CarbonPeriod::create($appointment->start_time, self::SLOT, $appointment->finish_time)->excludeEndDate();

        $hours = [];

        foreach ($period as $date) {
            $hours[] = $date->format('H:i');
        }
        return $hours;

    }

}
