<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Requests\Product\StoreProductRequest;
use App\Http\Requests\Product\UpdateProductRequest;
use App\Repositories\Product\ProductRepository;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\DB;

use App\Models\File;

class ProductController extends Controller
{
    private ProductRepository $productRepository;   

    public function __construct(ProductRepository $productRepository)
    {
        $this->productRepository = $productRepository;
    }

    public function index($limit = 10): JsonResponse
    {
        try {
            return response()->json([
                'error' => false,
                'data' => $this->productRepository->paginate($limit),
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage(),
            ]);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreProductRequest $request): JsonResponse
    {
        $data = $request->validated();

        try {

            DB::beginTransaction();

            $product = $this->productRepository->create($data);

            DB::commit();


            return response()->json([
                'error' => false,
                'data' => $product,
            ],
                Response::HTTP_CREATED
            );

         } catch(\Exception $e) {
            DB::rollBack();

            return response()->json([
                'error' => true,
                'message' => $e->getMessage(),
            ]);

        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateProductRequest $request, string $id): JsonResponse
    {
        try {
            $data = $request->validated();

            return response()->json([
                'error' => false,
                'data' => $this->productRepository->update($id, $data),
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage(),
            ]);

        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id): JsonResponse
    {
        try {
            DB::beginTransaction();

            $this->productRepository->delete($id);

            DB::commit();

            return response()->json(null, Response::HTTP_NO_CONTENT);

        } catch (\Exception $e) {

            DB::rollBack();

            return response()->json([
                'error' => true,
                'message' => $e->getMessage(),
            ]);
        }
    }

    public function getImage(File $file)
    {
        return    $this->productRepository->getImage($file);
    }

    public function deleteImage(File $file)
    {
        try {
            
            $return = $this->productRepository->deleteImage($file) ?  response()->json(null, Response::HTTP_NO_CONTENT) :  null;
            return $return;
        } catch(\Exception $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage(),
            ]);

        }
    }
}
