<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class AuthController extends Controller
{
    /**
     * Create User
     * @param Request $request
     * @return User
     */
    public function createUser(Request $request)
    {
        try {
            //Validated
            $validateUser = Validator::make($request->all(),
                [
                    'name' => 'required',
                    'lastname' => 'required',
                    'identification' => 'required',
                    'email' => 'required|email|unique:users,email',
                    'password' => 'required',
                ]);

            if ($validateUser->fails()) {
                return response()->json([
                    'status' => false,
                    'message' => 'validation error',
                    'errors' => $validateUser->errors(),
                ], 401);
            }

            $user = User::create([
                'name' => $request->name,
                'lastname' => $request->lastname,
                'identification' => $request->identification,
                'email' => $request->email,
                'password' => Hash::make($request->password),
            ]);

            return response()->json([
                'error' => false,
                'name' => $user->name,
                'lastname' => $user->name,
                'identification' => $user->identification,
                'username' => $user->username,
                'email' => $user->email,
                'mobile' => $user->mobile,
                'gender' => $user->gender,
                'permissions' => [],
                'roles' => $user->roles,
                'token' => $user->createToken("API_TOKEN")->plainTextToken,
            ], 200);

        } catch (\Throwable $th) {
            return response()->json([
                'status' => false,
                'message' => $th->getMessage(),
            ], 500);
        }
    }

    /**
     * Login The User
     * @param Request $request
     * @return User
     */
    public function loginUser(Request $request)
    {
        try {
            $validateUser = Validator::make($request->all(),
                [
                    'email' => 'required|email',
                    'password' => 'required',
                ]);

            if ($validateUser->fails()) {
                return response()->json([
                    'status' => false,
                    'message' => 'validation error',
                    'errors' => $validateUser->errors(),
                ], 401);
            }

            if (!Auth::attempt($request->only(['email', 'password']))) {
                return response()->json([
                    'status' => false,
                    'message' => __('auth.failed') // 'Email & Password does not match with our record.',
                ], 401);
            }

            $user = User::where('email', $request->email)->first();

            return response()->json([
                'error' => false,
                'name' => $user->name,
                'lastname' => $user->name,
                'identification' => $user->identification,
                'username' => $user->username,
                'email' => $user->email,
                'mobile' => $user->mobile,
                'gender' => $user->gender,
                'roles' => $user->roles,
                'occupation' => $user->Occupation,
                'hear_about_us' => $user->HearAboutUs,
                'token' => $user->createToken("API_TOKEN")->plainTextToken,
            ], 200);

        } catch (\Throwable $th) {
            return response()->json([
                'status' => false,
                'message' => $th->getMessage(),
            ], 500);
        }
    }

    public function logout(Request $request)
    {
        $user = Auth::user();
        $user->tokens()->delete();

        return response()->json(null);
    }

    public function checkStatus(Request $request)
    {
        $user = Auth::user();

        return response()->json([
            'error' => false,
            'name' => $user->name,
            'lastname' => $user->name,
            'username' => $user->username,
            'email' => $user->email,
            'mobile' => $user->mobile,
            'gender' => $user->gender,
            'permissions' => [],
            'roles' => $user->roles,
            'token' => $user->createToken("API_TOKEN")->plainTextToken,
        ], 200);
    }
}
